const {
  time,
  loadFixture,
} = require("@nomicfoundation/hardhat-toolbox/network-helpers");
const { anyValue } = require("@nomicfoundation/hardhat-chai-matchers/withArgs");
const { expect } = require("chai");

describe("Hyperreal", function () {
  let h, owner, team, advisor, community, partner, advisor2, community2, partner2, blockTimestamp;

  async function deploy() {
    [owner, team, advisor, community, partner, advisor2, community2, partner2] = await ethers.getSigners();
    const Hyperreal = await ethers.getContractFactory("Hyperreal");
    h = await Hyperreal.deploy(team.address);
    blockTimestamp = await time.latest();
  }

  describe("Deployment", async function () {
    it("Should deploy", async function () {
      await deploy();
      expect(await h.name()).to.eq('Hyperreal');
    });

    it("Check multisig balance", async function () {
      const target = ethers.parseEther('600000000');
      expect(await h.balanceOf(owner.address)).to.eq(target);
    });

    it("Check team balance", async function () {
      const target = ethers.parseEther('50000000');
      expect(await h.balanceOf(team.address)).to.eq(target);
    });

    it("Check contract balance", async function () {
      const target = ethers.parseEther('50000000');
      expect(await h.balanceOf(h.target)).to.eq(target);
    });

    it("Check timestamp", async function () {
      const ts = await h.lockTimestamp();
      expect(ts).to.gt(1692086645);
      expect(ts).to.lt(1786781045);
    });
  });

  describe("unlockTeamTokens", async function () {
    it("Revert when too early", async function () {
      await expect(h.unlockTeamTokens(team.address)).to.be.reverted;
    });

    it("Roll blockchain forwards", async function () {
      const ONE_YEAR_IN_SECS = 365 * 24 * 60 * 60;
      const unlockTime = blockTimestamp + ONE_YEAR_IN_SECS;
      await time.increaseTo(unlockTime);
    });

    it("Relock team tokens", async function () {
      await h.connect(owner).relockTeamTokens();
    });

    it("Revert again too early", async function () {
      await expect(h.unlockTeamTokens(team.address)).to.be.reverted;
    });

    it("Roll blockchain forwards", async function () {
      const TWO_YEAR_IN_SECS = 2 * 365 * 24 * 60 * 60;
      const unlockTime = blockTimestamp + TWO_YEAR_IN_SECS;
      await time.increaseTo(unlockTime);
    });


    it("Revert except for owner", async function () {
      await expect(h.connect(advisor).unlockTeamTokens(team.address)).to.be.reverted;
    });

    it("Should unlock team tokens", async function () {
      await h.connect(owner).unlockTeamTokens(team.address);
      const target = ethers.parseEther('100000000');
      expect(await h.balanceOf(team.address)).to.eq(target);
    });

    it("2nd unlock reverted", async function () {
      await expect(h.unlockTeamTokens(team.address)).to.be.reverted;
    });
  });

  describe("mintAdvisor", async function () {
    const amount = ethers.parseEther('50000000');
    it("Revert except for owner", async function () {
      await expect(h.connect(team).mintAdvisor(advisor.address, amount)).to.be.reverted;
    });

    it("Mint Advisor Tokens", async function () {
      await h.connect(owner).mintAdvisor(advisor.address, amount);
    });

    it("Check advisor balance", async function () {
      expect(await h.balanceOf(advisor.address)).to.eq(amount);
    });

    it("Check minted balance", async function () {
      expect(await h.advisorMinted()).to.eq(amount);
    });
    
    it("Revert if limit is exceeded", async function () {
      await expect(h.connect(team).mintAdvisor(advisor.address, amount+1n)).to.be.reverted;
    });
  });

  describe("mintLockedAdvisor", async function () {
    const amount = ethers.parseEther('50000000');

    it("Revert except for owner", async function () {
      blockTimestamp = await time.latest();
      const ONE_MONTH_IN_SECS = 30 * 24 * 60 * 60;
      const unlockTime = blockTimestamp + ONE_MONTH_IN_SECS;
      await expect(h.connect(team).mintLockedAdvisor(advisor2.address, amount, unlockTime)).to.be.reverted;
    });

    it("Revert on locking existing holder", async function () {
      blockTimestamp = await time.latest();
      const ONE_MONTH_IN_SECS = 30 * 24 * 60 * 60;
      const unlockTime = blockTimestamp + ONE_MONTH_IN_SECS;
      await expect(h.connect(owner).mintLockedAdvisor(advisor.address, amount, unlockTime)).to.be.reverted;
    });

    it("Mint Advisor Tokens", async function () {
      blockTimestamp = await time.latest();
      const ONE_MONTH_IN_SECS = 30 * 24 * 60 * 60;
      const unlockTime = blockTimestamp + ONE_MONTH_IN_SECS;
      await h.connect(owner).mintLockedAdvisor(advisor2.address, amount, unlockTime);
    });

    it("Check advisor balance", async function () {
      expect(await h.balanceOf(advisor2.address)).to.eq(amount);
    });

    it("Revert if limit is exceeded", async function () {
      blockTimestamp = await time.latest();
      const ONE_MONTH_IN_SECS = 30 * 24 * 60 * 60;
      const unlockTime = blockTimestamp + ONE_MONTH_IN_SECS;
      await expect(h.connect(team).mintLockedAdvisor(advisor2.address, amount+1n, unlockTime)).to.be.reverted;
    });

    it("Revert on send locked tokens", async function () {
      await expect(h.connect(advisor2).transfer(advisor.address, 1)).to.be.reverted;
    });

    it("Roll blockchain forwards", async function () {
      blockTimestamp = await time.latest();
      const ONE_MONTH_IN_SECS = 30 * 24 * 60 * 60;
      const unlockTime = blockTimestamp + ONE_MONTH_IN_SECS;
      await time.increaseTo(unlockTime);
    });

    it("Test transfer", async function () {
      h.connect(advisor2).transfer(advisor.address, 1);
    });
  });

  describe("mintCommunity", async function () {
    const amount = ethers.parseEther('50000000');
    it("Revert except for owner", async function () {
      await expect(h.connect(team).mintCommunity(community.address, amount)).to.be.reverted;
    });

    it("Mint Community Tokens", async function () {
      await h.connect(owner).mintCommunity(community.address, amount);
    });

    it("Check community balance", async function () {
      expect(await h.balanceOf(community.address)).to.eq(amount);
    });

    it("Check minted balance", async function () {
      expect(await h.communityMinted()).to.eq(amount);
    });
    
    it("Revert if limit is exceeded", async function () {
      await expect(h.connect(team).mintCommunity(community.address, amount+1n)).to.be.reverted;
    });
  });

  describe("mintLockedCommunity", async function () {
    const amount = ethers.parseEther('50000000');

    it("Revert except for owner", async function () {
      blockTimestamp = await time.latest();
      const ONE_MONTH_IN_SECS = 30 * 24 * 60 * 60;
      const unlockTime = blockTimestamp + ONE_MONTH_IN_SECS;
      await expect(h.connect(team).mintLockedCommunity(community2.address, amount, unlockTime)).to.be.reverted;
    });

    it("Revert on locking existing holder", async function () {
      blockTimestamp = await time.latest();
      const ONE_MONTH_IN_SECS = 30 * 24 * 60 * 60;
      const unlockTime = blockTimestamp + ONE_MONTH_IN_SECS;
      await expect(h.connect(owner).mintLockedCommunity(community.address, amount, unlockTime)).to.be.reverted;
    });

    it("Mint Community Tokens", async function () {
      blockTimestamp = await time.latest();
      const ONE_MONTH_IN_SECS = 30 * 24 * 60 * 60;
      const unlockTime = blockTimestamp + ONE_MONTH_IN_SECS;
      await h.connect(owner).mintLockedCommunity(community2.address, amount, unlockTime);
    });

    it("Check community balance", async function () {
      expect(await h.balanceOf(community2.address)).to.eq(amount);
    });

    it("Revert if limit is exceeded", async function () {
      blockTimestamp = await time.latest();
      const ONE_MONTH_IN_SECS = 30 * 24 * 60 * 60;
      const unlockTime = blockTimestamp + ONE_MONTH_IN_SECS;
      await expect(h.connect(team).mintLockedCommunity(community2.address, amount+1n, unlockTime)).to.be.reverted;
    });

    it("Revert on send locked tokens", async function () {
      await expect(h.connect(community2).transfer(community.address, 1)).to.be.reverted;
    });

    it("Roll blockchain forwards", async function () {
      blockTimestamp = await time.latest();
      const ONE_MONTH_IN_SECS = 30 * 24 * 60 * 60;
      const unlockTime = blockTimestamp + ONE_MONTH_IN_SECS;
      await time.increaseTo(unlockTime);
    });

    it("Test transfer", async function () {
      h.connect(community2).transfer(community.address, 1);
    });
  });

  describe("mintPartner", async function () {
    const amount = ethers.parseEther('50000000');
    it("Revert except for owner", async function () {
      await expect(h.connect(team).mintPartner(partner.address, amount)).to.be.reverted;
    });
  
    it("Mint Partner Tokens", async function () {
      await h.connect(owner).mintPartner(partner.address, amount);
    });
  
    it("Check partner balance", async function () {
      expect(await h.balanceOf(partner.address)).to.eq(amount);
    });
  
    it("Check minted balance", async function () {
      expect(await h.partnerMinted()).to.eq(amount);
    });
    
    it("Revert if limit is exceeded", async function () {
      await expect(h.connect(team).mintPartner(partner.address, amount+1n)).to.be.reverted;
    });
  });


  describe("mintLockedPartner", async function () {
    const amount = ethers.parseEther('50000000');

    it("Revert except for owner", async function () {
      blockTimestamp = await time.latest();
      const ONE_MONTH_IN_SECS = 30 * 24 * 60 * 60;
      const unlockTime = blockTimestamp + ONE_MONTH_IN_SECS;
      await expect(h.connect(team).mintLockedPartner(partner2.address, amount, unlockTime)).to.be.reverted;
    });

    it("Revert on locking existing holder", async function () {
      blockTimestamp = await time.latest();
      const ONE_MONTH_IN_SECS = 30 * 24 * 60 * 60;
      const unlockTime = blockTimestamp + ONE_MONTH_IN_SECS;
      await expect(h.connect(owner).mintLockedPartner(partner.address, amount, unlockTime)).to.be.reverted;
    });

    it("Mint Partner Tokens", async function () {
      blockTimestamp = await time.latest();
      const ONE_MONTH_IN_SECS = 30 * 24 * 60 * 60;
      const unlockTime = blockTimestamp + ONE_MONTH_IN_SECS;
      await h.connect(owner).mintLockedPartner(partner2.address, amount, unlockTime);
    });

    it("Check Partner balance", async function () {
      expect(await h.balanceOf(partner2.address)).to.eq(amount);
    });

    it("Revert if limit is exceeded", async function () {
      blockTimestamp = await time.latest();
      const ONE_MONTH_IN_SECS = 30 * 24 * 60 * 60;
      const unlockTime = blockTimestamp + ONE_MONTH_IN_SECS;
      await expect(h.connect(team).mintLockedPartner(partner2.address, amount+1n, unlockTime)).to.be.reverted;
    });

    it("Revert on send locked tokens", async function () {
      await expect(h.connect(partner2).transfer(partner.address, 1)).to.be.reverted;
    });

    it("Roll blockchain forwards", async function () {
      blockTimestamp = await time.latest();
      const ONE_MONTH_IN_SECS = 30 * 24 * 60 * 60;
      const unlockTime = blockTimestamp + ONE_MONTH_IN_SECS;
      await time.increaseTo(unlockTime);
    });

    it("Test transfer", async function () {
      h.connect(partner2).transfer(partner.address, 1);
    });
  });

});
