const { time } = require("@nomicfoundation/hardhat-network-helpers");
const { anyValue } = require("@nomicfoundation/hardhat-chai-matchers/withArgs");
const { expect } = require("chai");
const IERC20 = "@openzeppelin/contracts/token/ERC20/IERC20.sol:IERC20";

/*
  Note requires a mainnet fork to interact with lido/curve
  npx hardhat node --fork https://eth-mainnet.alchemyapi.io/v2/YOURAPIKEY
*/

describe("RastaKitty100", function () {
  let ras,pool,owner,user1,oneETH;
  

  it("Deployment", async function () {
    [owner,user1] = await ethers.getSigners();
    oneETH = ethers.utils.parseEther('1');
    const RAS = await hre.ethers.getContractFactory("RastaKitty100");
    ras = await RAS.deploy();
    await ras.deployed();
    expect(await ras.name()).to.be.oneOf(['TESTKITTY','RastaKitty']);
  });

  it("Add Liquidity", async function () {
    await ras.addLiquidity();
    const balance = await ras.balanceOf(ras.address);
    expect(balance).to.lt(10000);
  });

  it("Revert On Second addLiquidity", async function () {
    await expect(ras.addLiquidity()).to.be.reverted;
  });

  it("Renounce Ownership", async function () {
    await ras.renounceOwnership();
    const owner = await ras.owner();
    expect(owner).to.eq('0x0000000000000000000000000000000000000000');
  });

  it("Team Balance", async function () {
    const balance = await ras.balanceOf('0xaA63F6b94856A2f32333DfD116626efcCde74261');
    expect(balance).to.eq(0);
  });

  it("Check Liquidity Pool", async function () {
    const poolAddress = await ras.pool();
    const poolAbi = ["function fee() external view returns (uint24)","function swap(address,bool,int256,uint160,bytes) external returns (int256, int256)"];
    pool = await hre.ethers.getContractAt(poolAbi, poolAddress);
    const fee = await pool.fee();
    expect(fee).to.eq(3000);
  });


  it("Make Swap", async function () {
    const poolAddress = await ras.pool();
    const wethAbi = [
      "function name() view returns (string)",
      "function symbol() view returns (string)",
      "function decimals() view returns (uint8)",
      "function balanceOf(address) view returns (uint256)",
      "function transfer(address, uint256) returns (bool)",
      "function approve(address, uint256) returns (bool)",
      "function deposit() payable",
      "function withdraw(uint256)",
    ];
    const routerAddress = '0xE592427A0AEce92De3Edee1F18E0157C05861564';
    const wethAddress = '0xC02aaA39b223FE8D0A0e5C4F27eAD9083C756Cc2';
    const weth = await hre.ethers.getContractAt(wethAbi, wethAddress);
    const value = ethers.utils.parseEther('1');
    await weth.connect(owner).deposit({value});
    await weth.connect(owner).approve(routerAddress, value);
    const routerAbi = ['function exactInputSingle((address tokenIn, address tokenOut, uint24 fee, address recipient, uint256 deadline, uint256 amountIn, uint256 amountOutMinimum, uint160 sqrtPriceLimitX96)) external payable returns (uint256 amountOut)'];
    router = await hre.ethers.getContractAt(routerAbi, routerAddress);
    const deadline = Math.floor(Date.now() / 1000) + 600;
    const tx = await router.connect(owner).exactInputSingle([wethAddress, ras.address, 3000, owner.address, deadline, value, 0, 0]);
    await tx.wait();
  });
  
  it("owner Balance", async function () {
    const value = ethers.utils.parseEther('0.98');
    const balance = await ras.balanceOf(owner.address);
    const teamSupply = ethers.utils.parseEther('0');
    expect(balance).to.gt(value);
  });

  it("Team Balance", async function () {
    const value = ethers.utils.parseEther('0.009');
    const balance = await ras.balanceOf('0xaA63F6b94856A2f32333DfD116626efcCde74261');
    expect(balance).to.gt(value);
  });

  it("Deploy Multiple Times", async function () {
    // This is to check the fixOrdering works depending on deployment address
    for (let i = 0; i < 10; i++) {
      const RAS = await hre.ethers.getContractFactory("RastaKitty");
      const ras2 = await RAS.deploy();
      await ras2.deployed();
      await ras2.addLiquidity();
      const balance = await ras2.balanceOf(ras2.address);
      expect(balance).to.lt(10000);
    }
  });

});
